using UnityEngine;
using System.Collections;
using System.Runtime.InteropServices;

namespace hiveAdiz
{
	public class AdizNative
	{

		private static AdizNative instance = null;

		private AdizNative() {}

		~AdizNative() {
			//
		}

		private string adKey = "";

		public static AdizNative Initialize(BannerSize size, AdEventHandlers handler) {

			string defaultKey = AdizCallbackManager.defaultNativeKey;
			if(string.IsNullOrEmpty(defaultKey)) { 
				AdizLogger.d("adKey is null or empty.");
				return null; 
			}

			instance = new AdizNative();
			instance.internalInit(defaultKey, size, handler);
			return instance;
		}

		public static AdizNative Initialize(string hiveAdKey, BannerSize size, AdEventHandlers handler) {

			if(string.IsNullOrEmpty(hiveAdKey)) { 
				AdizLogger.d("adKey is null or empty.");
				return null; 
			}

			instance = new AdizNative();
			instance.internalInit(hiveAdKey, size, handler);
			return instance;
		}

		public static AdizNative InitializeWithPlacementId(string placementId, BannerSize size, AdEventHandlers handler) {

			if(string.IsNullOrEmpty(placementId)) { 
				AdizLogger.d("placementId is null or empty.");
				return null; 
			}

			string hiveAdKey = AdizUnityNativeCaller.Instance.GetHiveAdKeyFromPlacementId(placementId);
			if(string.IsNullOrEmpty(hiveAdKey)) { 
				AdizLogger.d("placementId is invalid");
				return null; 
			}

			instance = new AdizNative();
			instance.internalInit(hiveAdKey, size, handler);
			return instance;
		}

		private void release() {
			if(AdizCallbackManager.callbackHandler.ContainsKey(adKey)) {
				AdizCallbackManager.callbackHandler.Remove(adKey);	
			}

			instance = null;
		}

	#if UNITY_ANDROID && !UNITY_EDITOR
		
		private AndroidJavaObject currentActivity = null;
		private AndroidJavaObject hiveAdizPlugin = null;

		private void internalInit(string hiveAdKey, BannerSize size, AdEventHandlers handler) {
			AdizLogger.d("AdizNative Init "+hiveAdKey);

			AndroidJavaClass jc = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
			currentActivity = jc.GetStatic<AndroidJavaObject>("currentActivity");
			hiveAdizPlugin = new AndroidJavaObject("com.hive.adiz.plugin.AdizUnityNative");

			adKey = hiveAdKey;
			hiveAdizPlugin.Call("initialize", currentActivity, hiveAdKey, (int) size);
			
			if(AdizCallbackManager.callbackHandler.ContainsKey(hiveAdKey)) {
				AdizCallbackManager.callbackHandler.Remove(hiveAdKey);	
			}
			AdizCallbackManager.callbackHandler.Add(hiveAdKey, handler);
		}

		public void Load() {
			AdizLogger.d("AdizNative Load");
			hiveAdizPlugin.Call("load");
		}

		public void Show() {
			AdizLogger.d("AdizNative Show");
			hiveAdizPlugin.Call("show");
		}

		public void Show(PositionType position) {
			AdizLogger.d("AdizNative Show");
			int posNum = (int) position;
			hiveAdizPlugin.Call("show", posNum);
		}

		public void Destroy() {
			hiveAdizPlugin.Call("destroy");
			release();
		}

		public bool IsInitialized() {
			return hiveAdizPlugin.Call<bool>("isInitialized");
		}

		public bool IsLoaded() {
			return hiveAdizPlugin.Call<bool>("isLoaded");
		}

	#elif UNITY_IOS && !UNITY_EDITOR

		[DllImport("__Internal")]
		private static extern void Adiz_NativeInitialize(string hiveAdKey, int size);

		[DllImport("__Internal")]
		private static extern void Adiz_NativeLoad(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern void Adiz_NativeShow(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern void Adiz_NativeShowPosition(string hiveAdKey, int position);

		[DllImport("__Internal")]
		private static extern void Adiz_NativeDestroy(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern bool Adiz_NativeIsInitialized(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern bool Adiz_NativeIsLoaded(string hiveAdKey);

		private void internalInit(string hiveAdKey, BannerSize size, AdEventHandlers handler) {
			AdizLogger.d("AdizNative Init "+hiveAdKey);
			adKey = hiveAdKey;
			
			if(AdizCallbackManager.callbackHandler.ContainsKey(hiveAdKey)) {
				AdizCallbackManager.callbackHandler.Remove(hiveAdKey);	
			}
			AdizCallbackManager.callbackHandler.Add(hiveAdKey, handler);

			Adiz_NativeInitialize(hiveAdKey, (int) size);
		}

		public void Load() {
			AdizLogger.d("AdizNative Load");
			Adiz_NativeLoad(adKey);
		}

		public void Show() {
			AdizLogger.d("AdizNative Show");
			Adiz_NativeShow(adKey);
		}

		public void Show(PositionType position) {
			AdizLogger.d("AdizNative Show Position");
			Adiz_NativeShowPosition(adKey, (int) position);
		}

		public void Destroy() {
			Adiz_NativeDestroy(adKey);
			release();
		}

		public bool IsInitialized() {
			return Adiz_NativeIsInitialized(adKey);
		}

		public bool IsLoaded() {
			return Adiz_NativeIsLoaded(adKey);
		}
	#else

		private void internalInit(string hiveAdKey, BannerSize size, AdEventHandlers handler) {
			
		}

		public void Load() {
			
		}		

		public void Show() {
			
		}

		public void Show(PositionType position) {
			
		}

		public void Destroy() {
			
		}
		
		public bool IsInitialized() {
			return false;
		}

		public bool IsLoaded() {
			return false;
		}
	#endif
		
	}
}